// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.outbound.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.pay.echo.outbound.domain.OutboundEchoRequest;
import com.google.samples.pgp.PgpEncryptor;
import com.google.samples.pgp.PgpEncryptor.PgpDecryptionException;
import com.google.samples.pgp.PgpEncryptor.PgpEncryptionException;
import com.google.samples.pgp.PgpKeyManager;
import java.io.IOException;
import java.util.logging.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

@Service
public class OutboundEchoService {

  private static final Logger LOGGER = Logger.getLogger(OutboundEchoService.class.getName());
  private final RestTemplate restTemplate;
  private final ObjectMapper objectMapper;

  @Value("${google.hosted.echo.url}")
  private String googleHostedEchoUrl;

  @Value("${payment.integrator.account.id}")
  private String paymentIntegratorAccountId;

  public OutboundEchoService(RestTemplate restTemplate, ObjectMapper objectMapper) {
    this.restTemplate = restTemplate;
    this.objectMapper = objectMapper;
  }

  /**
   * POSTS a request to the Google Standard Payments API with the encrypted message. The decrypted
   * response from the POST request is returned.
   *
   * @see <a href=
   *     "https://developers.google.com/standard-payments/payment-integrator-diagnostic-api/rest/v2/TopLevel/echo">
   *     Echo Api </a>
   * @param message The message to be relayed to the payments API.
   * @return The decrypted response.
   */
  public ResponseEntity<String> postBillPayNotificationEchoApi(String message) {
    try {
      PgpEncryptor pgpEncryptor = new PgpEncryptor(PgpKeyManager.getInstance());
      String jsonEchoRequestMessage = objectMapper.writerWithDefaultPrettyPrinter()
              .writeValueAsString(createEchoRequestWithMessage(message));
      LOGGER.info("Decrypted Request Data: " + jsonEchoRequestMessage);
      String encryptedMessage = pgpEncryptor.encrypt(jsonEchoRequestMessage);
      String decryptedData =
          pgpEncryptor.decrypt(postStandardPaymentsEchoApi(encryptedMessage).getBody());
      LOGGER.info("Decrypted Response Data: " + decryptedData);
      return new ResponseEntity<>(decryptedData, HttpStatus.ACCEPTED);
    } catch (PgpEncryptionException pgpEncryptionException) {
      return new ResponseEntity<>(pgpEncryptionException.getMessage(), HttpStatus.BAD_REQUEST);
    } catch (PgpDecryptionException | IOException exception) {
      return new ResponseEntity<>(exception.getMessage(), HttpStatus.NOT_FOUND);
    }
  }

  /**
   * Sends a POST request to the Standard Payments Echo API.
   *
   * @param encryptedMessage The encrypted message.
   * @return A HTTP entity response containing the encrypted message.
   */
  private HttpEntity<String> postStandardPaymentsEchoApi(String encryptedMessage) {
    ResponseEntity<String> httpEntityResponse =
        restTemplate.postForEntity(
            UriComponentsBuilder.newInstance()
                .scheme("https")
                .host(googleHostedEchoUrl)
                .path(paymentIntegratorAccountId)
                .build().toUriString(), buildRequestEntity(encryptedMessage), String.class);
    LOGGER.info("Response Code: " + httpEntityResponse.getStatusCode()
            + " Encrypted Response Data: " + httpEntityResponse.getBody());
    return httpEntityResponse;
  }

  /**
   * Creates the request with the encrypted payload as an octet stream.
   *
   * @param encryptedMessage The encrypted message.
   * @return A HTTP entity containing the encrypted message.
   */
  private HttpEntity<String> buildRequestEntity(String encryptedMessage) {
    HttpHeaders headers = new HttpHeaders();
    headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
    return new HttpEntity<>(encryptedMessage, headers);
  }

  /**
   * Creates the echo request.
   *
   * @param message The message to be added to the request.
   * @return The echo request.
   */
  public OutboundEchoRequest createEchoRequestWithMessage(String message) {
    return OutboundEchoRequest.builder()
        .setClientMessage(message)
        .build();
  }
}
