// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.inbound.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.pay.echo.inbound.domain.InboundEchoRequest;
import com.google.pay.echo.inbound.domain.InboundEchoResponse;
import com.google.samples.pgp.PgpEncryptor;
import com.google.samples.pgp.PgpEncryptor.PgpDecryptionException;
import com.google.samples.pgp.PgpEncryptor.PgpEncryptionException;
import com.google.samples.pgp.PgpKeyManager;
import java.io.IOException;
import java.util.logging.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

@Service
public class InboundEchoService {

  private static final Logger LOGGER = Logger.getLogger(InboundEchoService.class.getName());

  private final ObjectMapper objectMapper;

  public InboundEchoService(ObjectMapper objectMapper) {
    this.objectMapper = objectMapper;
  }

  /**
   * POST endpoint called by Google to test connectivity. The request is decrypted, and an encrypted
   * response is returned.
   *
   * @see <a href=
   *     "https://developers.google.com/standard-payments/payment-integrator-diagnostic-api/v2/TopLevel/echo">
   *     Echo Api </a>
   * @param echoRequest The encrypted request coming from Google.
   * @return An encrypted string.
   */
  public ResponseEntity<String> decryptRequestAndEncryptEchoResponse(String echoRequest) {
    try {
      LOGGER.info("Encrypted Request from Google: " + echoRequest);
      PgpEncryptor pgpEncryptor = new PgpEncryptor(PgpKeyManager.getInstance());
      String decryptedRequest = pgpEncryptor.decrypt(echoRequest);
      LOGGER.info("Decrypted Request from Google: " + decryptedRequest);
      String echoResponseString =
          objectMapper.writeValueAsString(convertEchoRequestStringToEchoResponse(decryptedRequest));
      LOGGER.info("Decrypted Response to Google: " + echoResponseString);
      String encryptedResponse = pgpEncryptor.encrypt(echoResponseString);
      LOGGER.info("Encrypted Response to Google: " + encryptedResponse);
      return new ResponseEntity<>(encryptedResponse, HttpStatus.ACCEPTED);
    } catch (PgpEncryptionException pgpEncryptionException) {
      return new ResponseEntity<>(pgpEncryptionException.getMessage(), HttpStatus.BAD_REQUEST);
    } catch (PgpDecryptionException | IOException exception) {
      return new ResponseEntity<>(exception.getMessage(), HttpStatus.NOT_FOUND);
    }
  }

  /**
   * Converts the echo response from the raw decrypted request string.
   *
   * @param decryptedRequest The decrypted request body coming from Google.
   * @return The echo response.
   */
  private InboundEchoResponse convertEchoRequestStringToEchoResponse(String decryptedRequest) {
    try {
      return googleEchoRequestToEchoResponse(objectMapper.readValue(decryptedRequest, InboundEchoRequest.class));
    } catch (IOException exception) {
      LOGGER.info("IO Exception Converting Echo Request " + exception.getMessage());
      throw new RuntimeException(exception);
    }
  }

  /**
   * Creates the echo response by parsing the data from the echo request.
   *
   * @param inboundEchoRequest The request body coming from Google.
   * @return The echo response.
   */
  private InboundEchoResponse googleEchoRequestToEchoResponse(InboundEchoRequest inboundEchoRequest) {
    return InboundEchoResponse.builder()
        .setClientMessage(inboundEchoRequest.getClientMessage())
        .build();
  }
}
