// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.inbound.domain;

/**
 * Defines the echo response returned from "/v2/echo".
 *
 * @see <a href=
 *     "https://developers.google.com/standard-payments/payment-integrator-diagnostic-api/v2/TopLevel/echo#response-body">
 *     Echo Response </a>
 */
public class InboundEchoResponse {

  private InboundResponseHeader responseHeader;
  private String clientMessage;
  private String serverMessage;

  public InboundEchoResponse(Builder echoResponseBuilder) {
    this.responseHeader = echoResponseBuilder.responseHeader;
    this.clientMessage = echoResponseBuilder.clientMessage;
    this.serverMessage = echoResponseBuilder.serverMessage;
  }

  public static Builder builder() {
    return new InboundEchoResponse.Builder()
        .setClientMessage("Empty Client Message")
        .setServerMessage("Hello From Bank Little Bear!")
        .setResponseHeader(new InboundResponseHeader(new Timestamp()));
  }

  public String getClientMessage() {
    return clientMessage;
  }

  public void setClientMessage(String clientMessage) {
    this.clientMessage = clientMessage;
  }

  public String getServerMessage() {
    return serverMessage;
  }

  public void setServerMessage(String serverMessage) {
    this.serverMessage = serverMessage;
  }

  public InboundResponseHeader getResponseHeader() {
    return responseHeader;
  }

  public void setResponseHeader(InboundResponseHeader responseHeader) {
    this.responseHeader = responseHeader;
  }

  public static class Builder {

    private InboundResponseHeader responseHeader;
    private String clientMessage;
    private String serverMessage;

    public Builder setResponseHeader(InboundResponseHeader responseHeader) {
      this.responseHeader = responseHeader;
      return this;
    }

    public Builder setClientMessage(String clientMessage) {
      this.clientMessage = clientMessage;
      return this;
    }

    public Builder setServerMessage(String serverMessage) {
      this.serverMessage = serverMessage;
      return this;
    }

    public InboundEchoResponse build() {
      return new InboundEchoResponse(this);
    }
  }
}
