/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview This module serves routes that are non-specific to the add-on.
 * These routes include company homepage and sign-in related pages.
 */

var express = require('express');
var router = express.Router();
var checkAuthenticated = require('../middleware/checkAuthenticated');

/**
 * Route: /
 *
 * This is a page that is not part of the Classroom add-on. This page renders
 * if the user is attempting to visit the EdTEch company homepage.
 */
router.get('/', checkAuthenticated, function (req, res, next) {
  res.render('index', {
    title: 'Education Technology',
    message: 'Welcome to our website!'
  });
});

/**
 * Route: /failed
 *
 * The route is rendered if the sign-in process fails.
 */
router.get('/failed', function (req, res, next) {
  res.render('index', {
    title: 'Failed',
    message: 'You failed to log in.',
  });
});

/**
 * Route: /signin
 *
 * Renders a page with a button that the user must click to sign-in. This button
 * is required because we want to open the sign-in flow inside a pop-up window
 * and the action to click the button will prevent a pop-up blocker from
 * blocking the pop-up.
 */
router.get('/signin', function (req, res, next) {
  res.render('signin', {
    title: 'Sign-in',
  });
});

/**
 * Route: /logout
 *
 * Clears the session so that the user is no longer signed-in, then redirects
 * the user to a sign-in page.
 */
router.get('/logout', function (req, res, next) {
  req.session.user = null;
  req.session.tokens = null;
  res.redirect('/signin');
});

/**
 * Route: /closepopup
 *
 * Renders a page that does two things:
 * 1. Changes the location of the iframe window to the page the user tried to
 *    view before having to sign-in.
 * 2. Closes the pop-up window.
 */
router.get('/closepopup', function (req, res, next) {
  res.render('closepopup', {
    redirectUrl: '/classroom-addon'
  });
});

module.exports = router;
